# --------------------------------
# Hugh Jackovich & Lluc Cardoner |
# Python! @ TU Wien              |
# June 6, 2017                   |
# --------------------------------
# Sudoku board puzzle game       |
# --------------------------------
import numpy
import random
import colors   # For PyCharm users for printing colored text output

def new_block():
    """ Creates a new 3x3 array block and returns it """
    block = numpy.arange(1,10)
    block = numpy.random.permutation(block) # Randomizes the location of the numbers
    block = block.reshape(3,3)
    return block

def print_game(board, valid_targets):
    """ Prints the game board in a formatted manner """
    size = numpy.shape(board)
    dash = "-"
    
    print("#" + dash * 19 + "#")

    #Enable for colored output, but doesn't work in Idle shell (mostly PyCharm)
    """
    for row in range(size[0]):
        if row % 3 == 0:
            if row != 0:
                print(dash * (20))
        for column in range(len(board[row])):
            if column % 3 == 0:
                print("|", end=" ")
            if (row,column) in valid_targets:
                print(colors.red(board[row][column]), end=" ")
            else:
                print(board[row][column], end=" ")
        print("|")
    """

    # Uses line variables for printing
    for row in range(size[0]):
        line = ""
        if row % 3 == 0:
            if row == 0:
                pass
            else:
                print(dash * 20)
                
        for column in range(len(board[row])):
            if column % 3 == 0:
                line += "|"
            line += str(board[row][column]) + " "
        line = line.rstrip(" ")
        print(line + "|")
    
    print("#" + dash * 19 + "#\n")

def test_valid(board):
    """ Tests current board state for validity """
    valid = True

    # Checks all rows in the board
    for row in board:
        test_length = len(set(row).difference([0])) # Number of elements in row different to 0
        nonzeros = numpy.count_nonzero(row)         # Counts all non-zero elements in the row
        if test_length < nonzeros:
            valid = False
            break
        
    # board.T is the transpose of the puzzle, cycling through its columns. Checks all columns in the board
    for col in board.T:
        test_length = len(set(col).difference([0])) # Number of elements in col different to 0
        nonzeros = numpy.count_nonzero(col)         # Counts all non-zero elements in the column
        if test_length < nonzeros:
            valid = False
            break
        
    return valid

def generate_game(game_board = None, new = True):
    """ Generates the game board """
    available = set(range(1,10))

    # Fills in first 3 rows and first 3 columns of game_board
    if new:
        game_board = new_block()

        valid_row = False

        # Will create a row until it is valid
        while not valid_row:
            # Generates a row by double appending game_board with two new_blocks()
            new_row = numpy.append(game_board, new_block(), 1)
            new_row = numpy.append(new_row, new_block(), 1)
            if test_valid(new_row):
                valid_row = True

        valid_col = False
        
        # Will create a column until it is valid
        while not valid_col:
            new_col = numpy.append(game_board, new_block(), 0)
            new_col = numpy.append(new_col, new_block(), 0)
            if test_valid(new_col):
                new_col = numpy.append(new_col[3:],numpy.zeros((6,6), int), 1)  # Appends the new_col and fills rest of the board with zeros
                valid_col = True

        game_board = numpy.append(new_row, new_col, 0)  # Final append of the first rows and columns

    # Fills in rest of game_board
    valid = False
    while not valid:
        game_board[3:6, 3:6] = new_block()  # Slicing to replace desired partition of the board
        if test_valid(game_board[:6, :6]):  # Will test rows and columns up to the 6th position
            valid = True

    valid = False
    while not valid:
        game_board[6:, 6:] = new_block()    # Slicing to replace desired partition of the board
        if test_valid(game_board):          # Will test the whole board
            valid = True

    # Loops through all parts of the board except first 3 rows & columns
    for number in range(3,9):
        for number2 in range(3,9):

            # If game_board @ position is 0, replace it
            if game_board[number, number2] == 0:
                set1 = set(game_board[number])      # Creates set of numbers in current row
                set2 = set(game_board[:, number2])  # Creates set of numbers in current column
                available_set = available.difference(set1).difference(set2)    # Creates subset of only what is left of available numbers

                # If available_set length is 1, adds available number to the game_board from available_set
                if len(available_set) == 1:
                    game_board[number, number2] = available_set.pop()
                else:                               # Else resets part of the game_board and retries making
                    game_board[3:, 3:] = 0
                    return generate_game(game_board, False)
                
    return game_board

def check_input(target, choice, valid_targets):
    """ Checks if user input is valid for the puzzle """
    # Valid row and column selection 0-8 after input reduction
    if 9 < target[0] > 0 and 9 < target[1] > 0:
        print("Invalid target selection")
        return False
    
    # Valid number choice 1-9
    if 9 < choice < 0:
        print("Invalid choice number")
        return False
    
    # Check if the target is a valid target
    if target not in valid_targets:
        print("You cannot change the starting values numbers")
        return False
    
    return True

def insert_target(board, target, choice):
    """ Inserts the choice into the targeted position on the board """
    board[target[0]][target[1]] = choice
    return board

def make_puzzle(board, number):
    """ Selects random unique rows and columns and changes their value to 0"""
    valid_targets = []      # Will keep a list of random but unqiue chosen positions

    # Loops until the number of targets to remove, is equal to the amount removed
    while len(valid_targets) != number:
        number1 = random.randint(0, 8)
        number2 = random.randint(0, 8)
        target = (number1, number2)

        # If random target hasn't already been chosen, otherwise gets a new random target
        if target not in valid_targets:
            board[number1][number2] = 0
            valid_targets.append(target)
            
    return board, valid_targets     # Returns both the puzzle board and valid_target list

def is_finished(solution, board):
    """ Compares if the puzzle is equal to the solution """
    if numpy.array_equal(solution, board):
        return True
    
    return False

def main():
    """ Main function, handles flow of the program and checking for valid inputs """
    solution = generate_game(True)  # Generates the solution board
    sudoku = solution.copy()        # Copies solution, to then be manipulated

    # Prompts user to enter difficulty by removing X amount of numbers
    number = input("How many numbers would you like to remove: ")
    # If invalid input is given, will continuously ask for number until proper input is given
    while not number.isdigit():
        number = input("How many numbers would you like to remove: ")
    number = int(number)
    
    sudoku, valid_targets = make_puzzle(sudoku, number) # Changes sudoku board into actual puzzle to be solved

    # Loops until the sudoku board is solved    
    while not is_finished(solution, sudoku):
        print_game(sudoku, valid_targets)
        
        selection = input("Enter (row,column) number or s for solution: ").split()

        # Checks if user was entering valid input or prompting for solution to be printed
        if selection[0] == 's':
            print_game(solution, valid_targets)
        else:                                       # Else handles checking valid input
            targ = selection[0].split(',')
            row = targ[0][1:]
            column = targ[1][:-1]
            choice = selection[1]

            # Checks if all values are digits (integers) and above 0
            if row.isdigit() and column.isdigit() and choice.isdigit():
                target = (int(row)-1, int(column)-1)    # Forms target into tuple
                choice = int(selection[1])
            else:
                print("Enter valid input with format: (row,column) number")
                continue                                # Redirects flow back to the top of while loop
        if check_input(target, choice, valid_targets):
            sudoku = insert_target(sudoku, target, choice)

        print()                                         # Formatting

    print_game(sudoku)                                  # Prints solved puzzle
    print("Congrats you solved the puzzle!")    

if __name__ == '__main__':
    main()
